<?php
// ajax/orders_ajax.php
ini_set('display_errors', 0);
error_reporting(E_ALL);

require_once __DIR__ . '/../config/database.php';
header('Content-Type: application/json; charset=utf-8');

if (session_status() === PHP_SESSION_NONE) { session_start(); }

$user_id = $_SESSION['user_id'] ?? 0;
$group_id = $_SESSION['group_id'] ?? 0;
$action = $_REQUEST['action'] ?? '';

switch ($action) {
    case 'get_orders_list':
        handle_get_orders_list($conn, $user_id, $group_id);
        break;
    case 'delete_order':
        handle_delete_order($conn, $user_id);
        break;
    case 'get_order_details':
        handle_get_order_details($conn);
        break;
    default:
        echo json_encode(['status' => 'error', 'message' => 'Hành động không hợp lệ.']);
        break;
}

function handle_get_order_details($conn)
{
    $order_id = intval($_POST['order_id'] ?? 0);
    if ($order_id <= 0) { echo json_encode(['status' => 'error', 'message' => 'ID đơn hàng không hợp lệ.']); exit(); }

    // Lấy thông tin đơn
    $stmt = $conn->prepare("SELECT * FROM cms_orders WHERE id = ?");
    $stmt->bind_param("i", $order_id);
    $stmt->execute();
    $order = $stmt->get_result()->fetch_assoc();
    $stmt->close();

    if (!$order) { echo json_encode(['status' => 'error', 'message' => 'Không tìm thấy đơn hàng.']); exit(); }

    // Lấy chi tiết JSON
    $items = json_decode($order['detail_order'], true) ?: [];
    
    // --- LOGIC FIX LỖI HIỂN THỊ: JOIN NGƯỢC VỀ BẢNG SẢN PHẨM ---
    // Gom tất cả Product ID
    $product_ids = [];
    foreach($items as $item) {
        if(isset($item['id'])) $product_ids[] = intval($item['id']);
    }
    
    // Lấy thông tin gốc từ bảng products (để bù đắp nếu JSON thiếu)
    $products_map = [];
    if(!empty($product_ids)) {
        $ids_str = implode(',', $product_ids);
        $res = $conn->query("SELECT ID, prd_name, prd_code, prd_image_url FROM cms_products WHERE ID IN ($ids_str)");
        while($row = $res->fetch_assoc()) {
            $products_map[$row['ID']] = $row;
        }
    }
    
    // Làm giàu dữ liệu items
    $enriched_items = [];
    foreach($items as $item) {
        $pid = intval($item['id'] ?? 0);
        $db_info = $products_map[$pid] ?? null;
        
        $enriched_items[] = [
            'id' => $pid,
            // Nếu trong JSON không có code/name thì lấy từ DB
            'code' => $item['code'] ?? ($db_info['prd_code'] ?? 'N/A'),
            'name' => $item['name'] ?? ($db_info['prd_name'] ?? 'Sản phẩm đã xóa'),
            'image' => $db_info['prd_image_url'] ?? '',
            // Xử lý số lượng và giá (chấp nhận cả key 'qty' và 'quantity')
            'quantity' => floatval($item['quantity'] ?? $item['qty'] ?? 0),
            'price' => floatval($item['price'] ?? 0),
            'total' => floatval($item['quantity'] ?? $item['qty'] ?? 0) * floatval($item['price'] ?? 0)
        ];
    }

    echo json_encode([
        'status' => 'success', 
        'order' => $order, 
        'items' => $enriched_items // Trả về dữ liệu đã được xử lý kỹ
    ]);
    exit();
}

function handle_get_orders_list($conn, $user_id, $group_id)
{
    $draw = intval($_POST['draw'] ?? 0);
    $start = intval($_POST['start'] ?? 0);
    $length = intval($_POST['length'] ?? 10);

    // Map trạng thái
    $status_map = [
        0 => ['text' => 'Lưu tạm', 'class' => 'badge bg-warning text-dark'],
        1 => ['text' => 'Hoàn thành', 'class' => 'badge bg-success'],
        2 => ['text' => 'Đã hủy', 'class' => 'badge bg-danger'],
    ];

    $base_sql = "FROM cms_orders o 
                 LEFT JOIN cms_customers c ON o.customer_id = c.ID 
                 LEFT JOIN cms_stores s ON o.store_id = s.id 
                 LEFT JOIN cms_tables t ON o.table_id = t.id"; // Join thêm bảng Bàn
    
    $params = [];
    $types = "";
    $where_conditions = ["o.deleted = 0"]; // Mặc định không lấy đơn đã xóa mềm

    // Filter
    if (!empty($_POST['code'])) { $where_conditions[] = "o.output_code LIKE ?"; $params[] = "%{$_POST['code']}%"; $types .= "s"; }
    
    $filter_status = $_POST['filter_status'] ?? 'all';
    if ($filter_status === 'done') $where_conditions[] = "o.order_status = 1";
    if ($filter_status === 'draft') $where_conditions[] = "o.order_status = 0";
    if ($filter_status === 'debt') $where_conditions[] = "o.lack > 0";

    if (!empty($_POST['from'])) { $where_conditions[] = "DATE(o.sell_date) >= ?"; $params[] = $_POST['from']; $types .= "s"; }
    if (!empty($_POST['to'])) { $where_conditions[] = "DATE(o.sell_date) <= ?"; $params[] = $_POST['to']; $types .= "s"; }

    $where_sql = " WHERE " . implode(" AND ", $where_conditions);
    
    // Tổng quan
    $stmt_summary = $conn->prepare("SELECT COUNT(o.id) as total_orders, SUM(o.total_money) as total_amount, SUM(o.lack) as total_debt " . $base_sql . $where_sql);
    if (!empty($types)) $stmt_summary->bind_param($types, ...$params);
    $stmt_summary->execute();
    $summary = $stmt_summary->get_result()->fetch_assoc();
    $stmt_summary->close();
    
    // Đếm tổng
    $total_records = $conn->query("SELECT COUNT(ID) FROM cms_orders WHERE deleted = 0")->fetch_row()[0] ?? 0;

    // Lấy dữ liệu
    $sql_data = "SELECT o.ID as id, o.output_code, o.sell_date, o.total_money, o.lack, o.order_status, o.total_quantity,
                 COALESCE(c.customer_name, 'Khách lẻ') as customer_name, 
                 s.store_name, 
                 t.table_name 
                 " . $base_sql . $where_sql . " ORDER BY o.sell_date DESC LIMIT ?, ?";
    
    $params[] = $start; $params[] = $length; $types .= 'ii';
    
    $stmt_data = $conn->prepare($sql_data);
    $stmt_data->bind_param($types, ...$params);
    $stmt_data->execute();
    $data_rows = $stmt_data->get_result()->fetch_all(MYSQLI_ASSOC);
    $stmt_data->close();
    
    $data = [];
    foreach ($data_rows as $row) {
        // Xử lý hiển thị
        $status_html = "<span class='{$status_map[$row['order_status']]['class']}'>{$status_map[$row['order_status']]['text']}</span>";
        
        // Nút Sửa: Chuyển hướng sang POS kèm ID
        $edit_url = "/quanlybanhang/modules/pos/pos.php?edit_order_id=" . $row['id'];
        $btn_edit = '<a href="'.$edit_url.'" class="btn btn-sm btn-info" title="Sửa đơn"><i class="fas fa-pen"></i></a>';
        
        // Nút In
        $print_url = "/quanlybanhang/modules/pos/print_invoice.php?order_id=" . $row['id'];
        $btn_print = '<a href="'.$print_url.'" target="_blank" class="btn btn-sm btn-secondary" title="In bill"><i class="fas fa-print"></i></a>';
        
        // Nút Xóa
        $btn_delete = '<button class="btn btn-sm btn-danger btn-delete" data-id="'.$row['id'].'"><i class="fas fa-trash"></i></button>';

        $data[] = [
            "id" => $row['id'],
            "output_code" => '<span class="text-primary fw-bold">'.$row['output_code'].'</span>',
            "sell_date" => date('d/m/Y H:i', strtotime($row['sell_date'])),
            "customer_name" => $row['customer_name'],
            "table_name" => $row['table_name'] ? '<span class="badge bg-info">'.$row['table_name'].'</span>' : '-',
            "order_status" => $status_html,
            "total_money" => number_format($row['total_money']),
            "lack" => ($row['lack'] > 0) ? '<span class="text-danger fw-bold">'.number_format($row['lack']).'</span>' : '0',
            "actions" => '<div class="btn-group">'.$btn_edit.$btn_print.$btn_delete.'</div>'
        ];
    }
    
    echo json_encode([
        "draw" => $draw,
        "recordsTotal" => $total_records,
        "recordsFiltered" => $summary['total_orders'] ?? 0,
        "data" => $data,
        "summary" => $summary
    ]);
    exit();
}

function handle_delete_order($conn, $user_id) {
    $id = intval($_POST['order_id'] ?? 0);
    if($id <= 0) { echo json_encode(['status'=>'error','message'=>'ID lỗi']); exit; }
    
    $conn->query("UPDATE cms_orders SET deleted=1, user_upd=$user_id, updated=NOW() WHERE id=$id");
    // Xóa luôn sổ cái
    $conn->query("DELETE FROM cms_ledger WHERE source_type='ORDER' AND ref_id=$id");
    
    echo json_encode(['status'=>'success','message'=>'Đã xóa đơn hàng']);
    exit;
}
?>