<?php
require_once __DIR__ . '/../core/init.php';
header('Content-Type: application/json; charset=utf-8');

// Lấy action từ request và thông tin người dùng
$action = $_REQUEST['action'] ?? 'get_orders_list';
$user_id = $_SESSION['user_id'] ?? 0;

// Sử dụng switch-case để quản lý các action một cách rõ ràng
switch ($action) {
    case 'get_orders_list':
        handle_get_orders_list($conn);
        break;

    case 'delete_order':
        handle_delete_order($conn, $user_id);
        break;

    default:
        echo json_encode(['status' => 'error', 'message' => 'Hành động không hợp lệ.']);
        break;
}

/**
 * ===================================================================
 * HÀM XỬ LÝ VIỆC XÓA (HỦY) ĐƠN HÀNG
 * ===================================================================
 */
function handle_delete_order($conn, $user_id) {
    if ($user_id <= 0) {
        echo json_encode(['status' => 'error', 'message' => 'Bạn chưa đăng nhập.']);
        exit();
    }
    
    $order_id = intval($_POST['order_id'] ?? 0);

    if ($order_id <= 0) {
        echo json_encode(['status' => 'error', 'message' => 'ID đơn hàng không hợp lệ.']);
        exit();
    }

    $conn->begin_transaction();
    try {
        $stmt_get_order = $conn->prepare("SELECT detail_order, order_status FROM cms_orders WHERE ID = ? FOR UPDATE");
        $stmt_get_order->bind_param("i", $order_id);
        $stmt_get_order->execute();
        $order = $stmt_get_order->get_result()->fetch_assoc();
        $stmt_get_order->close();

        if (!$order) { throw new Exception("Không tìm thấy đơn hàng."); }
        if ($order['order_status'] == 0) { throw new Exception("Đơn hàng này đã được hủy từ trước."); }

        $items = json_decode($order['detail_order'], true);
        if (is_array($items) && !empty($items)) {
            $stmt_update_stock = $conn->prepare("UPDATE cms_products SET prd_sls = prd_sls + ? WHERE ID = ?");
            foreach ($items as $item) {
                $quantity = floatval($item['quantity'] ?? 0);
                $product_id = intval($item['id'] ?? 0);
                if ($quantity > 0 && $product_id > 0) {
                    $stmt_update_stock->bind_param("di", $quantity, $product_id);
                    $stmt_update_stock->execute();
                }
            }
            $stmt_update_stock->close();
        }

        $stmt_cancel_order = $conn->prepare("UPDATE cms_orders SET order_status = 0, user_upd = ?, updated = NOW() WHERE ID = ?");
        $stmt_cancel_order->bind_param("ii", $user_id, $order_id);
        $stmt_cancel_order->execute();
        $stmt_cancel_order->close();

        $conn->commit();
        echo json_encode(['status' => 'success', 'message' => 'Đã hủy đơn hàng và cập nhật lại tồn kho thành công!']);

    } catch (Exception $e) {
        $conn->rollback();
        echo json_encode(['status' => 'error', 'message' => 'Lỗi: ' . $e->getMessage()]);
    }
    exit();
}


/**
 * ===================================================================
 * HÀM XỬ LÝ VIỆC LẤY DANH SÁCH ĐƠN HÀNG (ĐÃ SỬA LỖI)
 * ===================================================================
 */
function handle_get_orders_list($conn) {
    $draw = intval($_POST['draw'] ?? 0);
    $start = intval($_POST['start'] ?? 0);
    $length = intval($_POST['length'] ?? 10);
    $order_column_index = intval($_POST['order'][0]['column'] ?? 1);
    $order_dir = $_POST['order'][0]['dir'] ?? 'desc';
    $columns = [0 => 'o.output_code', 1 => 'o.sell_date', 2 => 'c.customer_name', 3 => 'u.username', 4 => 'o.total_money', 5 => 'o.lack'];
    $order_by_column = $columns[$order_column_index] ?? 'o.ID';
    
    $status_map = [ 0 => ['text' => 'Đã hủy', 'class' => 'bg-danger'], 1 => ['text' => 'Hoàn thành', 'class' => 'bg-success']];
    $base_sql = "FROM cms_orders o LEFT JOIN cms_customers c ON o.customer_id = c.ID LEFT JOIN cms_users u ON o.user_init = u.ID";
    $params = []; $types = ""; $where_conditions = [];
    
    $view_mode = $_POST['view_mode'] ?? 'all';
    if ($view_mode == 'deleted') {
        $where_conditions[] = "o.order_status = 0";
    } elseif ($view_mode == 'debt') {
        $where_conditions[] = "o.order_status = 1 AND o.lack > 0";
    } else {
        // Mặc định của bạn là xem các đơn chưa bị xóa logic (deleted = 0). Nhưng vì chúng ta dùng order_status, 
        // có thể bạn muốn xem cả đơn Hoàn thành và Đã hủy khi chọn 'Tất cả'
        // Nếu chỉ muốn xem đơn Hoàn thành, hãy bỏ dấu chú thích dòng dưới
        // $where_conditions[] = "o.order_status = 1";
    }

    if (!empty($_POST['code'])) { $where_conditions[] = "o.output_code LIKE ?"; $params[] = "%{$_POST['code']}%"; $types .= "s"; }
    if (!empty($_POST['from'])) { $where_conditions[] = "o.sell_date >= ?"; $params[] = "{$_POST['from']} 00:00:00"; $types .= "s"; }
    if (!empty($_POST['to'])) { $where_conditions[] = "o.sell_date <= ?"; $params[] = "{$_POST['to']} 23:59:59"; $types .= "s"; }
    
    $where_sql = count($where_conditions) > 0 ? " WHERE " . implode(" AND ", $where_conditions) : "";

    // Lấy tổng số bản ghi
    $totalRecordsResult = $conn->query("SELECT COUNT(ID) FROM cms_orders");
    $recordsTotal = $totalRecordsResult ? $totalRecordsResult->fetch_row()[0] : 0;
    
    $queryFiltered = "SELECT COUNT(o.ID) as total " . $base_sql . $where_sql;
    $stmtFiltered = $conn->prepare($queryFiltered);
    if ($stmtFiltered) {
        if (!empty($types)) { $stmtFiltered->bind_param($types, ...$params); }
        $stmtFiltered->execute();
        $recordsFiltered = $stmtFiltered->get_result()->fetch_assoc()['total'] ?? 0;
        $stmtFiltered->close();
    } else { $recordsFiltered = 0; }

    // Lấy dữ liệu cho trang hiện tại - Đảm bảo SELECT đúng cột
    $sql_data = "SELECT o.ID, o.output_code, o.sell_date, o.total_money, o.lack, o.order_status, o.detail_order, COALESCE(c.customer_name, 'Khách lẻ') as customer_name, u.username " . $base_sql . $where_sql . " ORDER BY $order_by_column $order_dir LIMIT ?, ?";
    $data_params = $params; array_push($data_params, $start, $length);
    $data_types = $types . 'ii';
    
    $stmt_data = $conn->prepare($sql_data);
    $stmt_data->bind_param($data_types, ...$data_params);
    $stmt_data->execute();
    $result_set = $stmt_data->get_result()->fetch_all(MYSQLI_ASSOC);
    $stmt_data->close();
    
    // Logic kiểm tra khả năng trả hàng
    $order_ids = array_column($result_set, 'ID');
    $returned_quantities = [];
    if (!empty($order_ids)) {
        $ids_placeholder = implode(',', array_fill(0, count($order_ids), '?'));
        $return_sql = "SELECT order_id, product_id, SUM(quantity) as total_returned FROM cms_canreturn WHERE order_id IN ($ids_placeholder) GROUP BY order_id, product_id";
        $stmt_returns = $conn->prepare($return_sql);
        $stmt_returns->bind_param(str_repeat('i', count($order_ids)), ...$order_ids);
        $stmt_returns->execute();
        $returns_result = $stmt_returns->get_result();
        while ($row = $returns_result->fetch_assoc()) {
            $returned_quantities[$row['order_id']][$row['product_id']] = $row['total_returned'];
        }
        $stmt_returns->close();
    }

    $data = [];
    foreach($result_set as $order) {
        $can_return_order = false;
        $original_items = json_decode($order['detail_order'], true);
        if (is_array($original_items) && !empty($original_items)) {
            foreach($original_items as $item) {
                $returned_qty = $returned_quantities[$order['ID']][$item['id']] ?? 0;
                if ($item['quantity'] > $returned_qty) {
                    $can_return_order = true;
                    break;
                }
            }
        }
        
        $status_info = $status_map[$order['order_status']] ?? ['text' => 'N/A', 'class' => 'badge-secondary'];
        
        $return_button = ($can_return_order && $order['order_status'] == 1) ? '<a href="/quanlybanhang/modules/returns/create.php?order_id='.$order['ID'].'" class="text-info ms-2" title="Trả hàng"><i class="fas fa-undo"></i></a>' : '';
        $delete_button = ($order['order_status'] == 1) ? '<a href="#" class="text-danger ms-2 btn-delete-order" data-id="'.$order['ID'].'" title="Hủy đơn hàng"><i class="fas fa-trash"></i></a>' : '';

        $actions = '
            <a href="/quanlybanhang/modules/pos/pos.php?edit_order_id='.$order['ID'].'" class="text-warning" title="Sửa"><i class="fas fa-edit"></i></a>
            <a href="/quanlybanhang/modules/orders/print_invoice.php?order_id='.$order['ID'].'" class="text-secondary ms-2" target="_blank" title="In phiếu"><i class="fas fa-print"></i></a>
            '.$return_button.$delete_button;

        $data[] = [
            "ID" => $order['ID'],
            "output_code" => $order['output_code'],
            "sell_date" => date('d/m/Y H:i', strtotime($order['sell_date'])),
            "customer_name" => htmlspecialchars($order['customer_name']),
            "username" => htmlspecialchars($order['username'] ?? 'N/A'),
            "total_money" => $order['total_money'],
            "lack" => $order['lack'],
            "order_status" => "<span class='badge {$status_info['class']}'>{$status_info['text']}</span>",
            "actions" => $actions
        ];
    }
    
    // Lấy summary từ lần query trước, không cần query lại
    $summary = ["total_orders" => $recordsFiltered, "total_amount" => $summary_row['total_amount'] ?? 0, "total_debt" => $summary_row['total_debt'] ?? 0];
    
    echo json_encode([
        "draw" => $draw, 
        "recordsTotal" => intval($recordsTotal), 
        "recordsFiltered" => intval($recordsFiltered), 
        "data" => $data,
        "summary" => $summary
    ]);
    exit();
}
?>