<?php
require_once __DIR__ . '/../core/init.php';
header('Content-Type: application/json; charset=utf-8');

// Bắt đầu session nếu nó chưa được bắt đầu
if (session_status() == PHP_SESSION_NONE) {
    session_start();
}

// --- ĐỌC DỮ LIỆU JSON TỪ REQUEST ---
$json_data = file_get_contents('php://input');
$data = json_decode($json_data, true);

if ($data === null) {
    echo json_encode(['status' => 'error', 'message' => 'Lỗi đọc dữ liệu JSON từ request.']);
    exit();
}

// --- KIỂM TRA ĐẦU VÀO ---
$action = $data['action'] ?? '';
if ($action !== 'process_return') {
    echo json_encode(['status' => 'error', 'message' => 'Hành động không hợp lệ.']);
    exit();
}

// Lấy các dữ liệu cần thiết từ payload và SESSION
$user_id = $_SESSION['user_id'] ?? 0;
$store_id = $_SESSION['store_id'] ?? 0;

$order_id = intval($data['order_id'] ?? 0);
$customer_id = intval($data['customer_id'] ?? 0);
$items = $data['items'] ?? [];
$notes = trim($data['notes'] ?? '');

if ($order_id <= 0 || empty($items) || !is_array($items) || $user_id <= 0 || $store_id <= 0) {
    echo json_encode(['status' => 'error', 'message' => 'Dữ liệu đầu vào không hợp lệ hoặc bạn chưa đăng nhập (thiếu user_id hoặc store_id).']);
    exit();
}

$conn->begin_transaction();

try {
    // 1. Lấy thông tin đơn hàng gốc
    $stmt_order = $conn->prepare("SELECT output_code, detail_order, lack FROM cms_orders WHERE ID = ? FOR UPDATE");
    if ($stmt_order === false) throw new Exception("Lỗi SQL[Lấy đơn hàng]: " . $conn->error);
    $stmt_order->bind_param("i", $order_id);
    $stmt_order->execute();
    $order = $stmt_order->get_result()->fetch_assoc();
    $stmt_order->close();

    if (!$order) {
        throw new Exception("Không tìm thấy đơn hàng gốc.");
    }
    $original_items_list = json_decode($order['detail_order'], true);
    if (!is_array($original_items_list)) $original_items_list = [];
    $original_items = array_column($original_items_list, null, 'id');
    
    // 2. Lấy số lượng đã trả trước đó
    $stmt_returned = $conn->prepare("SELECT product_id, SUM(quantity) as total_returned FROM cms_canreturn WHERE order_id = ? GROUP BY product_id");
    if ($stmt_returned === false) throw new Exception("Lỗi SQL[Lấy hàng đã trả]: " . $conn->error);
    $stmt_returned->bind_param("i", $order_id);
    $stmt_returned->execute();
    $already_returned = [];
    $returned_result = $stmt_returned->get_result();
    while ($row = $returned_result->fetch_assoc()) {
        $already_returned[$row['product_id']] = $row['total_returned'];
    }
    $stmt_returned->close();
    
    $total_return_amount_by_sale_price = 0;

    // 3. Chuẩn bị các câu lệnh SQL
    $stmt_insert_return = $conn->prepare("INSERT INTO cms_canreturn (store_id, order_id, product_id, quantity, price, user_init, created) VALUES (?, ?, ?, ?, ?, ?, NOW())");
    if ($stmt_insert_return === false) throw new Exception("Lỗi SQL[INSERT cms_canreturn]: " . $conn->error);
    
    $stmt_update_inventory = $conn->prepare("INSERT INTO cms_inventory (store_id, product_id, quantity) VALUES (?, ?, ?) ON DUPLICATE KEY UPDATE quantity = quantity + VALUES(quantity)");
    if ($stmt_update_inventory === false) throw new Exception("Lỗi SQL[UPDATE cms_inventory]: " . $conn->error);

    // 4. Lặp qua các sản phẩm trả
    foreach ($items as $item_data) {
        $product_id = intval($item_data['id'] ?? 0);
        $return_qty = floatval($item_data['quantity'] ?? 0);

        if ($return_qty > 0 && $product_id > 0) {
            $original_qty = $original_items[$product_id]['quantity'] ?? 0;
            $returned_qty_before = $already_returned[$product_id] ?? 0;
            
            if (($return_qty + $returned_qty_before) > $original_qty) {
                throw new Exception("Số lượng trả của sản phẩm {$product_id} vượt quá số lượng đã mua.");
            }
            
            $sale_price = floatval($item_data['price']);

            // *** SỬA LỖI CÚ PHÁP TẠI ĐÂY ***
            // Cung cấp đúng 6 kiểu dữ liệu cho 6 dấu hỏi (?)
            // i=integer, d=double/float
            $stmt_insert_return->bind_param("iiiddi", $store_id, $order_id, $product_id, $return_qty, $sale_price, $user_id);
            $stmt_insert_return->execute();
            
            // Cập nhật tồn kho chi tiết
            $stmt_update_inventory->bind_param("iid", $store_id, $product_id, $return_qty);
            $stmt_update_inventory->execute();
            
            $total_return_amount_by_sale_price += $return_qty * $sale_price;
        }
    }
    $stmt_insert_return->close();
    $stmt_update_inventory->close();

    // 5. Cập nhật lại công nợ trên ĐƠN HÀNG GỐC
    if ($total_return_amount_by_sale_price > 0) {
        $stmt_update_order = $conn->prepare("UPDATE cms_orders SET lack = GREATEST(0, lack - ?) WHERE ID = ?");
        if ($stmt_update_order === false) throw new Exception("Lỗi SQL[UPDATE cms_orders]: " . $conn->error);
        $stmt_update_order->bind_param("di", $total_return_amount_by_sale_price, $order_id);
        $stmt_update_order->execute();
        $stmt_update_order->close();
    }
    
    // 6. Cập nhật tổng công nợ cho KHÁCH HÀNG
    if ($customer_id > 0 && $total_return_amount_by_sale_price > 0) {
        $stmt_update_customer_debt = $conn->prepare("UPDATE cms_customers SET total_debt = total_debt - ? WHERE ID = ?");
        if ($stmt_update_customer_debt === false) throw new Exception("Lỗi SQL[UPDATE cms_customers]: " . $conn->error);
        $stmt_update_customer_debt->bind_param("di", $total_return_amount_by_sale_price, $customer_id);
        $stmt_update_customer_debt->execute();
        $stmt_update_customer_debt->close();
    }
    
    // Đã loại bỏ hoàn toàn logic tạo phiếu chi
    
    $conn->commit();
    echo json_encode(['status' => 'success', 'message' => 'Xử lý trả hàng thành công! Tồn kho và công nợ đã được cập nhật.']);

} catch (Exception $e) {
    $conn->rollback();
    echo json_encode(['status' => 'error', 'message' => 'Đã xảy ra lỗi: ' . $e->getMessage()]);
}
exit();
?>