<?php
// FILE: ajax_create_mobile_order.php (Ví dụ)

ini_set('display_errors', 1); // Bật hiển thị lỗi để debug
error_reporting(E_ALL);

require_once __DIR__ . '/../core/init.php';
header('Content-Type: application/json; charset=utf-8');
if (session_status() === PHP_SESSION_NONE) { session_start(); }

$user_id = $_SESSION['user_id'] ?? 0;
if ($user_id === 0) {
    http_response_code(401);
    echo json_encode(['status' => 'error', 'message' => 'Phiên làm việc đã hết hạn. Vui lòng đăng nhập lại.']);
    exit();
}

// --- BẢO VỆ CHỐNG GỬI ĐƠN 2 LẦN (giữ nguyên) ---
$submitted_token = $_POST['order_token'] ?? '';
$session_token = $_SESSION['order_token'] ?? null;
if (empty($submitted_token) || empty($session_token) || $submitted_token !== $session_token) {
    http_response_code(409);
    echo json_encode(['status' => 'error', 'message' => 'Lỗi phiên tạo đơn, đơn hàng có thể đã được tạo. Vui lòng tải lại trang.']);
    exit();
}
unset($_SESSION['order_token']); // Hủy token ngay sau khi kiểm tra

$conn->begin_transaction();
try {
    // --- 1. LẤY DỮ LIỆU ĐẦU VÀO ---
    $customer_id = intval($_POST['customer_id'] ?? 1);
    $items_json = $_POST['cart_items'] ?? '[]';
    $items = json_decode($items_json, true);
    
    // Lấy thêm các thông tin thanh toán (nếu có từ giao diện mobile)
    $discount_value = floatval($_POST['discount_value'] ?? 0);
    $discount_type = $_POST['discount_type'] ?? 'fixed'; // 'fixed' hoặc 'percent'
    $vat_percent = floatval($_POST['vat'] ?? 0);
    $customer_pay = floatval($_POST['customer_pay'] ?? 0);

    if (empty($items) || !is_array($items)) {
        throw new Exception("Giỏ hàng trống hoặc dữ liệu không hợp lệ.");
    }

    // --- 2. KIỂM TRA TỒN KHO VÀ LẤY GIÁ VỐN ---
    $sub_total = 0;
    $total_origin_price = 0;
    $total_quantity = 0;
    $items_with_cost = [];

    foreach ($items as $item) {
        $p_id = intval($item['id']);
        $p_qty = floatval($item['quantity']);
        $p_price = floatval($item['price']);

        $stmt_prod = $conn->prepare("SELECT prd_name, prd_sls, IF(prd_average_cost > 0, prd_average_cost, prd_origin_price) as cost FROM cms_products WHERE ID = ? FOR UPDATE");
        $stmt_prod->bind_param("i", $p_id);
        $stmt_prod->execute();
        $prod_data = $stmt_prod->get_result()->fetch_assoc();
        $stmt_prod->close();

        if (!$prod_data) {
            throw new Exception("Sản phẩm '{$item['name']}' không tồn tại.");
        }
        if ($p_qty > floatval($prod_data['prd_sls'])) {
            throw new Exception("Sản phẩm '{$prod_data['prd_name']}' không đủ tồn kho (còn {$prod_data['prd_sls']}).");
        }

        $p_cost = floatval($prod_data['cost']);
        $sub_total += $p_price * $p_qty;
        $total_origin_price += $p_cost * $p_qty;
        $total_quantity += $p_qty;
        
        $item['origin_price'] = $p_cost; // Cập nhật giá vốn vào mảng
        $items_with_cost[] = $item;
    }

    // --- 3. TÍNH TOÁN GIÁ TRỊ CUỐI CÙNG CỦA ĐƠN HÀNG ---
    $discount_item_calc = ($discount_type === 'fixed') ? $discount_value : ($sub_total * ($discount_value / 100));
    $grand_total = $sub_total - $discount_item_calc;
    $vat_money = $grand_total * ($vat_percent / 100);
    $grand_total += $vat_money;
    
    // Nếu không có thông tin khách trả, mặc định là họ trả đủ
    if ($customer_pay == 0) {
        $customer_pay = $grand_total;
    }
    $lack = $grand_total - $customer_pay;

    // --- 4. LƯU ĐƠN HÀNG VỚI ĐẦY ĐỦ THÔNG TIN ---
    $store_id = $_SESSION['store_id'] ?? 0;
    $last_id_result = $conn->query("SELECT ID FROM cms_orders ORDER BY ID DESC LIMIT 1");
    $output_code = 'MB' . str_pad(($last_id_result->fetch_row()[0] ?? 0) + 1, 6, '0', STR_PAD_LEFT); // 'MB' for Mobile
    $detail_order_json = json_encode($items_with_cost);
    
    $sql_order = "INSERT INTO cms_orders (output_code, store_id, customer_id, sell_date, total_price, total_money, total_origin_price, discount_item, discount_percent, vat, detail_order, user_init, order_status, total_quantity, customer_pay, lack) VALUES (?, ?, ?, NOW(), ?, ?, ?, ?, ?, ?, ?, ?, 1, ?, ?, ?)";
    $stmt_order = $conn->prepare($sql_order);
    $stmt_order->bind_param('siiddddddsiddd', 
        $output_code, $store_id, $customer_id, 
        $sub_total, $grand_total, $total_origin_price, 
        $discount_item_calc, ($discount_type === 'percent' ? $discount_value : 0), $vat_percent, 
        $detail_order_json, $user_id, $total_quantity, 
        $customer_pay, $lack
    );
    $stmt_order->execute();
    $new_order_id = $conn->insert_id;
    $stmt_order->close();

    // --- 5. TRỪ KHO ĐÚNG CÁCH (CẢ SỐ LƯỢNG VÀ GIÁ TRỊ) ---
    $stmt_update_product = $conn->prepare("UPDATE cms_products SET prd_sls = prd_sls - ?, prd_total_value = prd_total_value - ? WHERE ID = ?");
    foreach ($items_with_cost as $item) {
        $qty_to_deduct = floatval($item['quantity']);
        $value_to_deduct = $qty_to_deduct * floatval($item['origin_price']);
        $id_to_update = intval($item['id']);
        $stmt_update_product->bind_param("ddi", $qty_to_deduct, $value_to_deduct, $id_to_update);
        $stmt_update_product->execute();
    }
    $stmt_update_product->close();
    
    // --- 6. TÍNH LẠI GIÁ VỐN CHO CÁC SẢN PHẨM BỊ ẢNH HƯỞNG ---
    $stmt_recalc_avg = $conn->prepare("UPDATE cms_products SET prd_average_cost = IF(prd_sls > 0, prd_total_value / prd_sls, 0) WHERE ID = ?");
    foreach($items_with_cost as $item) {
        $stmt_recalc_avg->bind_param("i", $item['id']);
        $stmt_recalc_avg->execute();
    }
    $stmt_recalc_avg->close();

    $conn->commit();
    echo json_encode(['status' => 'success', 'message' => "Đơn hàng {$output_code} đã được tạo thành công!", 'order_id' => $new_order_id]);

} catch (Exception $e) {
    $conn->rollback();
    http_response_code(400);
    echo json_encode(['status' => 'error', 'message' => $e->getMessage()]);
}
?>